#include "stdafx.h"
#include "msPlugInImpl.h"
#include "msLib.h"

#include "chBinWriter.hpp"
#include "fsFileUtil.hpp"
#include "fsFileStream.hpp"

namespace
{

//=============================================================================
//	Chunk types
//=============================================================================
const chDefs::Name c_GFRG = chDefs::MakeName('G', 'F', 'R', 'G');
const chDefs::Name c_GVER = chDefs::MakeName('G', 'V', 'E', 'R');
const chDefs::Name c_NVER = chDefs::MakeName('N', 'V', 'E', 'R');
const chDefs::Name c_TVER = chDefs::MakeName('T', 'V', 'E', 'R');
const chDefs::Name c_MATR = chDefs::MakeName('M', 'A', 'T', 'R');
const chDefs::Name c_MBAS = chDefs::MakeName('M', 'B', 'A', 'S');
const chDefs::Name c_GIND = chDefs::MakeName('G', 'I', 'N', 'D');
const chDefs::Name c_NIND = chDefs::MakeName('N', 'I', 'N', 'D');
const chDefs::Name c_TIND = chDefs::MakeName('T', 'I', 'N', 'D');


void write_vec4(chWriter &writer, const msVec4 &vec4)
{
	writer.Write(vec4[0]);
	writer.Write(vec4[1]);
	writer.Write(vec4[2]);
	writer.Write(vec4[3]);
}

}	// end of namespace

BOOL APIENTRY DllMain( HANDLE hModule, 
                       DWORD  ul_reason_for_call, 
                       LPVOID lpReserved
					 )
{
    switch (ul_reason_for_call)
	{
		case DLL_PROCESS_ATTACH:
		case DLL_THREAD_ATTACH:
		case DLL_THREAD_DETACH:
		case DLL_PROCESS_DETACH:
			break;
    }
    return TRUE;
}



cMsPlugIn*
CreatePlugIn ()
{
    return new cPlugIn ();
}



cPlugIn::cPlugIn ()
{
    strcpy (szTitle, "Ultimate Ride (.mx)...");
}



cPlugIn::~cPlugIn ()
{
}



int
cPlugIn::GetType ()
{
    return cMsPlugIn::eTypeExport;
}



const char*
cPlugIn::GetTitle ()
{
    return szTitle;
}


int
cPlugIn::Execute (msModel *pModel)
{
    if (!pModel)
        return -1;

    //
    // check, if we have something to export
    //
    if (msModel_GetMeshCount (pModel) == 0)
    {
        ::MessageBox (NULL, "The model is empty!  Nothing exported!", "MilkShape 3D ASCII Export", MB_OK | MB_ICONWARNING);
        return 0;
    }

    //
    // choose filename
    //
    OPENFILENAME ofn;
    memset (&ofn, 0, sizeof (OPENFILENAME));
    
    char szFile[MS_MAX_PATH];
    char szFileTitle[MS_MAX_PATH];
    char szDefExt[32] = "txt";
    char szFilter[128] = "Ultimate Ride Models (*.mx)\0*.mx\0All Files (*.*)\0*.*\0\0";
    szFile[0] = '\0';
    szFileTitle[0] = '\0';

    ofn.lStructSize = sizeof (OPENFILENAME);
    ofn.lpstrDefExt = szDefExt;
    ofn.lpstrFilter = szFilter;
    ofn.lpstrFile = szFile;
    ofn.nMaxFile = MS_MAX_PATH;
    ofn.lpstrFileTitle = szFileTitle;
    ofn.nMaxFileTitle = MS_MAX_PATH;
    ofn.Flags = OFN_HIDEREADONLY | OFN_OVERWRITEPROMPT | OFN_PATHMUSTEXIST;
    ofn.lpstrTitle = "Export Ultimate Ride Model";

    if (!::GetSaveFileName (&ofn))
        return 0;

    //
    // export
    //
	std::string	file_name(szFile);
	fsFileUtil::FileCreate(file_name);

	fsFileStream export_file(file_name, fsFileStream::e_WriteOnly);
	export_file.WriteHeader();

	chBinWriter writer(export_file);

    int i, j;
    char szName[MS_MAX_NAME];

    for (i = 0; i < msModel_GetMeshCount (pModel); i++)
    {
        msMesh *pMesh = msModel_GetMeshAt (pModel, i);
        msMesh_GetName (pMesh, szName, MS_MAX_NAME);

		// Geometry fragment chunk
		writer.WriteChunkHeader(c_GFRG, 0, true);

		// Vertices
        msVec3 Vertex;
		writer.WriteChunkHeader(c_GVER, 0, false);
		writer.Write(envType::Int16(msMesh_GetVertexCount (pMesh)));
        for (j = 0; j < msMesh_GetVertexCount (pMesh); j++)
		{
            msVertex *pVertex = msMesh_GetVertexAt (pMesh, j);
            msVertex_GetVertex (pVertex, Vertex);
			writer.Write(Vertex[0]);
			writer.Write(Vertex[1]);
			writer.Write(Vertex[2]);
		}
		writer.FinishChunk();

		// Normals
        msVec3 Normal;
		writer.WriteChunkHeader(c_NVER, 0, false);
		writer.Write(envType::Int16(msMesh_GetVertexNormalCount (pMesh)));
        for (j = 0; j < msMesh_GetVertexNormalCount (pMesh); j++)
		{
            msMesh_GetVertexNormalAt (pMesh, j, Normal);
			writer.Write(Normal[0]);
			writer.Write(Normal[1]);
			writer.Write(Normal[2]);
		}
		writer.FinishChunk();

		// Texture coordinates
        msVec2 uv;
		writer.WriteChunkHeader(c_TVER, 0, false);
		writer.Write(envType::Int16(msMesh_GetVertexCount (pMesh)));
        for (j = 0; j < msMesh_GetVertexCount (pMesh); j++)
		{
            msVertex *pVertex = msMesh_GetVertexAt (pMesh, j);
            msVertex_GetTexCoords (pVertex, uv);
			writer.Write(pVertex->u);
			//writer.Write(pVertex->v);
			writer.Write(float(1.0f - pVertex->v));
		}
		writer.FinishChunk();

		// Material
		int mat_index = msMesh_GetMaterialIndex (pMesh);
        msMaterial *pMaterial = msModel_GetMaterialAt (pModel, mat_index);

		writer.WriteChunkHeader(c_MATR, 0, true);
		writer.WriteChunkHeader(c_MBAS, 0, false);

        msVec4 color;
        msMaterial_GetDiffuse (pMaterial, color);
		write_vec4(writer, color); 
        msMaterial_GetAmbient (pMaterial, color);
		write_vec4(writer, color); 
        msMaterial_GetSpecular (pMaterial, color);
		write_vec4(writer, color); 
        msMaterial_GetEmissive (pMaterial, color);
		write_vec4(writer, color); 

		// maybe the material's shininess is the same as specular power?
		writer.Write(0); // specular power

		// Only writing diffuse texture now
        char szTexture[MS_MAX_PATH];
        msMaterial_GetDiffuseTexture (pMaterial, szTexture, MS_MAX_PATH);

		if (*szTexture)
		{
			writer.Write(envType::Int8(1));
			char *tex_name = ::strrchr(szTexture, '\\');
			if (tex_name)
				writer.Write(tex_name + 1);
			else
			{
				tex_name = ::strrchr(szTexture, '/');
				if (tex_name)
					writer.Write(tex_name + 1);
				else
					writer.Write(szTexture);
			}
		}
		else writer.Write(envType::Int8(0));
		
		writer.FinishChunk();
		writer.FinishChunk();
	
		// Geometry indices
        word nIndices[3];
		writer.WriteChunkHeader(c_GIND, 0, false);
		writer.Write(envType::Int16(msMesh_GetTriangleCount (pMesh)));
        for (j = 0; j < msMesh_GetTriangleCount (pMesh); j++)
		{
            msTriangle *pTriangle = msMesh_GetTriangleAt (pMesh, j);
            msTriangle_GetVertexIndices (pTriangle, nIndices);

			writer.Write(envType::Int16(nIndices[0]));
			writer.Write(envType::Int16(nIndices[1]));
			writer.Write(envType::Int16(nIndices[2]));
			
		}
		writer.FinishChunk();

		// Normal indices
		writer.WriteChunkHeader(c_NIND, 0, false);
		writer.Write(envType::Int16(msMesh_GetTriangleCount (pMesh)));
        for (j = 0; j < msMesh_GetTriangleCount (pMesh); j++)
		{
            msTriangle *pTriangle = msMesh_GetTriangleAt (pMesh, j);

			writer.Write(envType::Int16(pTriangle->nNormalIndices[0]));
			writer.Write(envType::Int16(pTriangle->nNormalIndices[1]));
			writer.Write(envType::Int16(pTriangle->nNormalIndices[2]));
			
		}
		writer.FinishChunk();

		// Texture indices, same as vertex indices
		writer.WriteChunkHeader(c_TIND, 0, false);
		writer.Write(envType::Int16(0)); // texture layer 0
		writer.Write(envType::Int16(msMesh_GetTriangleCount (pMesh)));
        for (j = 0; j < msMesh_GetTriangleCount (pMesh); j++)
		{
            msTriangle *pTriangle = msMesh_GetTriangleAt (pMesh, j);
            msTriangle_GetVertexIndices (pTriangle, nIndices);

			writer.Write(envType::Int16(nIndices[0]));
			writer.Write(envType::Int16(nIndices[1]));
			writer.Write(envType::Int16(nIndices[2]));
			
		}
		writer.FinishChunk();

		writer.FinishChunk();	// end of GFRG chunk
    }


    // dont' forget to destroy the model
    msModel_Destroy (pModel);

    return 0;
}
